%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Galerkin Polynomial chaos expansion on L96 model
% of example 4.9
% Created by by John Harlim
% Last edited: March 20, 2018
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear all, close all

J = 40;
T = 100;
u0 = sin([1:J]*2*pi/J)';
dt = 0.05;
mode = 5;

% generate Smolyak sparse grid
D = 2; % dimension
L = 8; % Smolyak level
[Y w] = nwspgr(@ccfornwspgr, D, L);

% evaluate Legendre polynomials on Smolyak grid
N = 15;
Phi = zeros(size(Y,1),size(Y,2),N+1);
P1 = ones(size(Y,1),size(Y,2));
P2 = Y;
Phi(:,:,1) = P1;
Phi(:,:,2) = P2;
for j=2:N                
    Phi(:,:,j+1) = (2*(j-1)+1)/j*Y.*P2 - (j-1)/j*P1;
    P1 = P2;
    P2 = Phi(:,:,j+1);
end

% integrate the expansion coefficients
multiindex = generatemultiindex(N,D);
P = size(multiindex,2);


% precompute e1jk
e1jk = sparse(P,P);
for j=1:P
    for k=1:P
        e1jk(j,k) = w'*(Phi(:,1,2).*Phi(:,1,multiindex(1,j)+1).*Phi(:,1,multiindex(1,k)+1));
        if (e1jk(j,k) < 1E-15) 
            e1jk(j,k) = 0;
        end
    end
end



% precompute ei1j1k1, ei2j2k2
ei1j1k1 = zeros(P,P,P);
ei2j2k2 = zeros(P,P,P);
gamma = zeros(P,1);
for i = 1:P
    gamma(i) = 1/(2*multiindex(1,i)+1)/(2*multiindex(2,i)+1);
    for j=1:P
        for k=1:P
            ei1j1k1(i,j,k) = w'*(Phi(:,1,multiindex(1,i)+1).*Phi(:,1,multiindex(1,j)+1).*Phi(:,1,multiindex(1,k)+1));
            ei2j2k2(i,j,k) = w'*(Phi(:,2,multiindex(2,i)+1).*Phi(:,2,multiindex(2,j)+1).*Phi(:,2,multiindex(2,k)+1));
            if (ei1j1k1(i,j,k) < 1E-15) 
                ei1j1k1(i,j,k) = 0;                
            end
            if (ei2j2k2(i,j,k) < 1E-15) 
                ei2j2k2(i,j,k) = 0;                
            end
        end
    end
end

% constant term
F = sparse(P,1);
F(1) = 8;
F(2) = 1;

% linear coefficients
A = sparse(P,P);
for j=1:P
    for k=1:P
        if (multiindex(2,j)==multiindex(2,k))
            A(j,k) = e1jk(j,k)/5/(1/(2*multiindex(1,j)+1));
        end
    end
end
A = A + eye(P);

% quadratic coefficients
B = ei1j1k1.*ei2j2k2;

% initial condition
u = zeros(P,J,T+1);
u(1,:,1) = u0;


for i = 1:T
    k1 = galerkinL96rhs(A,B,F,u(:,:,i),gamma);
    y1 = u(:,:,i) + k1*dt/2;
    k2 = galerkinL96rhs(A,B,F,y1,gamma);
    y2 = u(:,:,i) + k2*dt/2;
    k3 = galerkinL96rhs(A,B,F,y2,gamma);
    y3 = u(:,:,i) + k3*dt;
    k4 = galerkinL96rhs(A,B,F,y3,gamma);
    u(:,:,i+1) = u(:,:,i) + (k1+2*k2+2*k3+k4)*dt/6;
end

ubar1 = squeeze(u(1,mode,:));
vbar1 = sum(repmat(gamma(2:P,1),1,T+1).*squeeze(u(2:P,mode,:).^2))';

grey = [.4 .4 .4];
plot([0:dt:dt*T],ubar1,'*','color',grey,'linewidth',2)
hold on
plot([0:dt:dt*T],ubar1+sqrt(vbar1),'*','color',grey)
plot([0:dt:dt*T],ubar1-sqrt(vbar1),'*','color',grey)
hold off
set(gca,'fontsize',16)
xlabel('time')
ylabel('u_5')

%save galerkinL96 ubar1 vbar1 dt

return

%%% to plot the MonteCarlo solution, run example49mc.m first

load montecarloL96
hold on
plot([0:dt:dt*T],ubar(mode,:),'k--','linewidth',2)
plot([0:dt:dt*T],ubar(mode,:)+sqrt(vbar(mode,:)),'k--')
plot([0:dt:dt*T],ubar(mode,:)-sqrt(vbar(mode,:)),'k--')
hold off

set(gca,'fontsize',16)
xlabel('t')
ylabel('u_5')

%print -depsc -r100 galerkinL96.eps

